
--[[
StrikeFire / Remote Ground Strike
v1  (2025-08-23)

Highlights:
- Clean structure and sectioned comments.
- Only [FIRE] marker to engage; all CEASE-related phrases removed.
- One round per target; de-duplicates targets within the marker radius.
- Per-family target budget (ammo cap): GMLRS=6, ATACMS=1, Iskander=2.
- Range check: GMLRS 8–160nm, ATACMS 8–160nm, 9K720 40–215nm.
- Spawning: one launcher per group; logistics in a merged group; marker is auto-cleared after spawn.
- Cleanup: place a marker with CLEAN to remove all units spawned by this script.

Usage:
1) Place [FIRE] (brackets optional) on the F10 map; the script scans within SEARCH_RADIUS meters for enemy
   ground & static units and assigns exactly 1 round per target. Marker auto-clears after processing.
2) Use H-*/K-* (or aliases) to spawn formations at the marker point; marker auto-clears after spawn.
3) Use CLEAN to remove all units spawned by this script.
]]

------------------------------------------------------------
-- Tunables
------------------------------------------------------------
local FIRE_MARKER_RAW   = "FIRE"      -- Base text for fire command (will be normalized)
local SEARCH_RADIUS     = 3000         -- Search radius around the marker (meters)
local FIRE_RADIUS       = 120          -- FireAtPoint dispersion radius (meters)
local MAX_BATTERY       = 24           -- Max simultaneous launchers to task in one cycle
local CYCLE_DELAY       = 0.2          -- Throttle tasking (seconds)
local RESTORE_ROE_SEC   = 3            -- Restore ROE after this many seconds
local NM_TO_M           = 1852
local DEBUG             = false        -- Show debug messages

-- Always 1 round per target (don’t change; logic is designed around 1 per target)
local ROUND_PER_TARGET  = 1

-- Family range definitions (NM)
local RANGES_NM = {
  HIMARS_GMLRS  = { min = 8,  max = 160  },
  HIMARS_ATACMS = { min = 8,  max = 160 },
  ISK           = { min = 40, max = 215 },
}

-- ★ Per-vehicle target budget (effective ammo cap; 1 round per target)
-- Requirement: ATACMS=1, GMLRS=6, Iskander=2
local TARGET_BUDGET_PER_FAMILY = {
  HIMARS_GMLRS  = 6,
  HIMARS_ATACMS = 1,
  ISK           = 2,
}

-- Recognized launchers (desc.typeName → family)
local LAUNCHER_TYPES = {
  -- Common HIMARS (examples: CHAP mod type names; adjust if your mod uses different names)
  ["CHAP_M142_GMLRS_M30"]    = "HIMARS_GMLRS",
  ["CHAP_M142_GMLRS_M31"]    = "HIMARS_GMLRS",

  -- HIMARS ATACMS
  ["CHAP_M142_ATACMS_M39A1"] = "HIMARS_ATACMS",
  ["CHAP_M142_ATACMS_M48"]   = "HIMARS_ATACMS",

  -- 9K720 (Iskander)
  ["CHAP_9K720_HE"]          = "ISK",
  ["CHAP_9K720_Cluster"]     = "ISK",
}

-- De-dup threshold for targets (within one marker search); points closer than this are considered the same target
local DEDUP_EPS = 10  -- meters

------------------------------------------------------------
-- Helpers
------------------------------------------------------------
local function say(msg, t)
  if DEBUG then trigger.action.outText(msg, t or 8) end
end

local function normText(s)
  -- Uppercase, trim spaces, strip square brackets (accepts both "FIRE" and "[FIRE]")
  local up = string.upper(s or "")
  up = up:gsub("%s+", "")
  up = up:gsub("[%[%]]", "")  -- strip []
  return up
end

local function d2(p1, p2)
  local dx, dz = p1.x - p2.x, p1.z - p2.z
  return math.sqrt(dx * dx + dz * dz)
end

-- Temporarily open ROE to ensure fire, then restore
local ROE_CACHE = {}  -- unitName -> lastROE
local function setROE_OpenFire(unit)
  local c = unit:getController(); if not c then return end
  if c.getOption then ROE_CACHE[unit:getName()] = c:getOption(0) end -- 0 = ROE
  if c.setOption then c:setOption(0, 2) end                          -- 2 = Open Fire
end
local function restoreROE(unit)
  local c = unit:getController(); if not c then return end
  local last = ROE_CACHE[unit:getName()]
  if last ~= nil and c.setOption then c:setOption(0, last) end
end

local function listFriendlyLaunchers(coalitionID, refPoint)
  local carriers = {}
  for _, g in ipairs(coalition.getGroups(coalitionID, Group.Category.GROUND) or {}) do
    if g and g:isExist() then
      for _, u in ipairs(g:getUnits() or {}) do
        if u and u:isExist() and u:isActive() then
          local desc = u:getDesc()
          local fam  = desc and desc.typeName and LAUNCHER_TYPES[desc.typeName]
          if fam then
            carriers[#carriers + 1] = {
              unit    = u,
              name    = u:getName(),
              family  = fam,
              dist    = d2(u:getPoint(), refPoint),
              budget  = TARGET_BUDGET_PER_FAMILY[fam] or 1, -- remaining target budget
            }
          end
        end
      end
    end
  end
  table.sort(carriers, function(a, b) return a.dist < b.dist end)
  return carriers
end

local function inFamilyRangeMeters(unit, family, targetPoint)
  local rng = RANGES_NM[family]
  if not rng then return true end
  local d = d2(unit:getPoint(), targetPoint)
  return d >= rng.min * NM_TO_M and d <= rng.max * NM_TO_M
end

-- FireAtPoint requires Vec2 {x=…, y=…} where y = world.z
local function schedulePushFireAtPoint(unit, tgt, delaySec)
  local vec2 = { x = tgt.x, y = tgt.z }
  local task = {
    id = 'FireAtPoint',
    params = {
      point = vec2,
      radius = FIRE_RADIUS,
      expendQty = ROUND_PER_TARGET,   -- 1 per target
      expendQtyEnabled = true,
    }
  }
  timer.scheduleFunction(function(param, time)
    local u = param.u
    if u and u:isExist() then
      local c = u:getController()
      if c then c:pushTask(task) end
    end
    return nil
  end, { u = unit }, timer.getTime() + (delaySec or 0))
end

-- Optional debug markers
local function debugMark(coal, pt, txt, life)
  if not DEBUG then return end
  local id = math.random(100000, 999999)
  trigger.action.markToCoalition(id, txt or "tgt", { x = pt.x, y = 0, z = pt.z }, coal)
  timer.scheduleFunction(function() trigger.action.removeMark(id) return nil end, nil, timer.getTime() + (life or 8))
end

-- De-dup targets: if a point is within DEDUP_EPS to an existing one, skip it
local function addUniqueTarget(tlist, pos)
  for _, p in ipairs(tlist) do
    if d2(p, pos) < DEDUP_EPS then return end
  end
  tlist[#tlist + 1] = pos
end

------------------------------------------------------------
-- Main flow ([FIRE] marker)
------------------------------------------------------------
local function handleMarker(event)
  -- Only handle add/change
  if event.id ~= world.event.S_EVENT_MARK_ADDED and event.id ~= world.event.S_EVENT_MARK_CHANGE then
    return
  end

  -- Ignore spectators
  local coal = event.coalition
  if not coal or (coal ~= coalition.side.BLUE and coal ~= coalition.side.RED) then
    return
  end

  -- Normalize marker text
  local raw = event.text or ""
  local key = normText(raw)
  local FIRE = normText(FIRE_MARKER_RAW)

  -- Only handle FIRE
  if key ~= FIRE then return end

  local enemy = (coal == coalition.side.BLUE) and coalition.side.RED or coalition.side.BLUE
  local p0 = event.pos
  say(string.format("[StrikeFire] marker@ (%.0f, %.0f)", p0.x, p0.z))

  -- 1) Search enemy ground & static around the marker (SEARCH_RADIUS), with de-dup
  local targets = {}
  for _, g in ipairs(coalition.getGroups(enemy, Group.Category.GROUND) or {}) do
    if g and g:isExist() then
      for _, u in ipairs(g:getUnits() or {}) do
        if u and u:isExist() then
          local pos = u:getPoint()
          if d2(pos, p0) <= SEARCH_RADIUS then
            addUniqueTarget(targets, pos)
          end
        end
      end
    end
  end
  for _, s in ipairs(coalition.getStaticObjects(enemy) or {}) do
    if s and s:isExist() then
      local pos = s:getPoint()
      if d2(pos, p0) <= SEARCH_RADIUS then
        addUniqueTarget(targets, pos)
      end
    end
  end

  trigger.action.outText(string.format("[FIRE] Found %d targets within %dm of the marker (1 round per target).", #targets, SEARCH_RADIUS), 10)
  if #targets == 0 then if event.idx then trigger.action.removeMark(event.idx) end; return end

  -- 2) Collect friendly launchers, sort by distance, cap by MAX_BATTERY; load their target budgets
  local carriers = listFriendlyLaunchers(coal, p0)
  if #carriers == 0 then
    trigger.action.outText("[FIRE] No available launchers (HIMARS/ATACMS/9K720) found.", 8)
    if event.idx then trigger.action.removeMark(event.idx) end
    return
  end

  local usable = {}
  for i, e in ipairs(carriers) do
    if i > MAX_BATTERY then break end
    if e.budget > 0 then
      setROE_OpenFire(e.unit)
      usable[#usable + 1] = e
    end
  end
  if #usable == 0 then
    trigger.action.outText("[FIRE] All available launchers have 0 remaining target budget.", 8)
    if event.idx then trigger.action.removeMark(event.idx) end
    return
  end

  -- 3) Round-robin assignment: for each target, pick the first qualified launcher in range with remaining budget
  local idx, assigned, skipped = 1, 0, 0
  local report, delay = {}, 0

  for _, tgt in ipairs(targets) do
    local ok = false
    local tries = 0
    debugMark(coal, tgt, "TGT")
    while tries < #usable do
      local shooter = usable[idx]
      if shooter and shooter.unit and shooter.unit:isExist() then
        if shooter.budget > 0 then
          if inFamilyRangeMeters(shooter.unit, shooter.family, tgt) then
            schedulePushFireAtPoint(shooter.unit, tgt, delay)  -- 1 per target
            shooter.budget = shooter.budget - 1                -- consume budget
            report[shooter.name] = (report[shooter.name] or 0) + 1
            assigned = assigned + 1
            ok = true
            delay = delay + CYCLE_DELAY
            idx = (idx % #usable) + 1
            break
          end
        end
      end
      idx = (idx % #usable) + 1
      tries = tries + 1
    end
    if not ok then skipped = skipped + 1 end
  end

  -- 4) Report & restore ROE
  local lines = { string.format("[FIRE] Assignment complete: %d targets; %d skipped.", assigned, skipped) }
  for _, e in ipairs(usable) do
    lines[#lines + 1] = string.format("  %s: executed %d, remaining %d/%d",
      e.name, report[e.name] or 0, e.budget, TARGET_BUDGET_PER_FAMILY[e.family] or 1)
  end
  trigger.action.outText(table.concat(lines, "\n"), 10)

  timer.scheduleFunction(function()
    for _, e in ipairs(usable) do
      if e.unit and e.unit:isExist() then restoreROE(e.unit) end
    end
  end, nil, timer.getTime() + RESTORE_ROE_SEC)

  if event.idx then trigger.action.removeMark(event.idx) end
end

-- Register event handler (fire)
world.addEventHandler({
  onEvent = function(self, event)
    if event.id == world.event.S_EVENT_MARK_ADDED
    or event.id == world.event.S_EVENT_MARK_CHANGE then
      handleMarker(event)
    end
  end
})


------------------------------------------------------------
-- Spawner (H-*/K-*; CLEAN clears)
------------------------------------------------------------
if not _G.STRIKE then STRIKE = {} end
STRIKE.SPAWNED_GROUPS = STRIKE.SPAWNED_GROUPS or {}

-- Formations (marker text → composition)
STRIKE.FORMATIONS = {
  -- USA / HIMARS
  ["H-BLK"] = { units = {
    { type = "CHAP_M142_GMLRS_M31",    num = 4 },
    { type = "CHAP_M1083",             num = 2 },
  }},
  ["H-SEAD"] = { units = {
    { type = "CHAP_M142_ATACMS_M39A1", num = 2 },
    { type = "CHAP_M142_GMLRS_M31",    num = 2 },
    { type = "CHAP_M1083",             num = 1 },
  }},
  ["H-DS"]  = { units = {
    { type = "CHAP_M142_ATACMS_M48",   num = 3 },
    { type = "CHAP_M142_ATACMS_M39A1", num = 1 },
    { type = "CHAP_M1083",             num = 2 },
  }},
  ["H-QRF"] = { units = {
    { type = "CHAP_M142_GMLRS_M31",    num = 2 },
    { type = "CHAP_M142_ATACMS_M48",   num = 1 },
    { type = "CHAP_M1083",             num = 1 },
  }},
  ["H-BRG"] = { units = {
    { type = "CHAP_M142_GMLRS_M31",    num = 6 },
    { type = "CHAP_M1083",             num = 4 },
  }},

  -- Russia / 9K720 (Iskander)
  ["K-HE"]  = { units = {
    { type = "CHAP_9K720_HE",          num = 3 },
  }},
  ["K-CL"]  = { units = {
    { type = "CHAP_9K720_Cluster",     num = 3 },
  }},
  ["K-MIX"] = { units = {
    { type = "CHAP_9K720_HE",          num = 2 },
    { type = "CHAP_9K720_Cluster",     num = 2 },
  }},
  ["K-DS"]  = { units = {
    { type = "CHAP_9K720_Cluster",     num = 3 },
    { type = "CHAP_9K720_HE",          num = 1 },
  }},
  ["K-BRG"] = { units = {
    { type = "CHAP_9K720_Cluster",     num = 4 },
    { type = "CHAP_9K720_HE",          num = 2 },
  }},
}

-- Legacy aliases (backward compatibility)
STRIKE.ALIAS = {
  ["HIMARS-BLOCK"]   = "H-BLK",
  ["HIMARS-SEAD"]    = "H-SEAD",
  ["HIMARS-STRIKE"]  = "H-DS",
  ["HIMARS-QRF"]     = "H-QRF",
  ["HIMARS-BARRAGE"] = "H-BRG",

  ["ISK-HE"] = "K-HE", ["ISK-CL"] = "K-CL", ["ISK-MIX"] = "K-MIX",
  ["ISK-DS"] = "K-DS", ["ISK-BRG"] = "K-BRG",

  ["9K-HE"]  = "K-HE", ["9K-CL"]  = "K-CL",  ["9K-MIX"] = "K-MIX",
  ["9K-DS"]  = "K-DS", ["9K-BRG"] = "K-BRG",
}

local function normKey(k) return STRIKE.ALIAS[k] or k end

-- Spawn management
local function trackGroup(name)
  STRIKE.SPAWNED_GROUPS[#STRIKE.SPAWNED_GROUPS + 1] = name
end

local function destroyGroupByName(name)
  local g = Group.getByName(name)
  if g and g:isExist() then g:destroy() end
end

function STRIKE.clearAll()
  for i = #STRIKE.SPAWNED_GROUPS, 1, -1 do
    destroyGroupByName(STRIKE.SPAWNED_GROUPS[i])
    table.remove(STRIKE.SPAWNED_GROUPS, i)
  end
  trigger.action.outText("Cleared all units spawned by this script.", 8)
end

local function isShooterType(t)
  local u = (t or ""):upper()
  return u:find("M142", 1, true) or u:find("9K720", 1, true)
end

-- Spawning: one launcher per group; logistics consolidated
local function spawnStrikeGroup(template, coalitionSide, center, markId)
  local countryId = (coalitionSide == coalition.side.RED) and country.id.CJTF_RED or country.id.CJTF_BLUE

  -- Formation params (avoid column; grid + jitter)
  local HEADING_DEG, MIN_SPACING, JITTER = 90, 150, 40
  local heading = math.rad(HEADING_DEG)
  local function rotate(x, z, a)
    local c, s = math.cos(a), math.sin(a)
    return x * c - z * s, x * s + z * c
  end

  -- Split: shooters / logistics
  local shooters, logist = {}, {}
  for _, e in ipairs(template.units) do
    -- Exclude MLRS (per requirement)
    if e.type ~= "MLRS" and e.type ~= "M270 MLRS" then
      if isShooterType(e.type) then
        shooters[#shooters + 1] = { type = e.type, num = e.num }
      else
        logist[#logist + 1]   = { type = e.type, num = e.num }
      end
    end
  end

  -- Layout calc
  local total = 0
  for _, e in ipairs(shooters) do total = total + e.num end
  for _, e in ipairs(logist)  do total = total + e.num end
  local cols = math.ceil(math.sqrt(total))
  local rows = math.ceil(total / cols)
  local placed = 0
  local function nextSlot()
    placed = placed + 1
    local r = math.floor((placed - 1) / cols)
    local c = (placed - 1) % cols
    local gx = (c - (cols - 1) / 2) * MIN_SPACING + (math.random() * 2 - 1) * JITTER
    local gz = (r - (rows - 1) / 2) * MIN_SPACING + (math.random() * 2 - 1) * JITTER
    local rx, rz = rotate(gx, gz, heading)
    return center.x + rx, center.z + rz
  end

  -- Shooters: one launcher per group
  for _, e in ipairs(shooters) do
    for i = 1, e.num do
      local gx, gz = nextSlot()
      local isISK = (e.type:upper():find("9K720", 1, true) ~= nil)
      local prefix = isISK and "ISK" or "HIMARS"
      local gName = string.format("%s_%04d", prefix, math.random(0, 9999))
      local gd = {
        visible = false, lateActivation = false, uncontrolled = false, hidden = false,
        task = "Ground Nothing", tasks = {}, x = gx, y = gz, name = gName, start_time = 0,
        units = { {
          type = e.type, name = gName .. "_U1", skill = "High", x = gx, y = gz,
          heading = heading + (math.random() * 0.6 - 0.3), transportable = { randomTransportable = false }
        } }
      }
      coalition.addGroup(countryId, Group.Category.GROUND, gd)
      trackGroup(gName)
    end
  end

  -- Logistics: merge into one group
  if #logist > 0 then
    local units = {}
    local base = string.format("LOGI_%04d", math.random(0, 9999))
    local n = 0
    for _, e in ipairs(logist) do
      for i = 1, e.num do
        local gx, gz = nextSlot()
        n = n + 1
        units[#units + 1] = {
          type = e.type, name = base .. "_U" .. n, skill = "Average", x = gx, y = gz,
          heading = heading + (math.random() * 0.6 - 0.3), transportable = { randomTransportable = false }
        }
      end
    end
    local gd = {
      visible = false, lateActivation = false, uncontrolled = false, hidden = false,
      task = "Ground Nothing", tasks = {}, x = center.x, y = center.z, name = base, start_time = 0,
      units = units,
    }
    coalition.addGroup(countryId, Group.Category.GROUND, gd)
    trackGroup(base)
  end

  trigger.action.outText("Formation spawned (one launcher per group).", 8)
  if markId then trigger.action.removeMark(markId) end
end

-- Marker events (spawn / clear)
local function onMapMarkerChange(event)
  if not event.text or event.text == "" then return end
  local text = string.upper(event.text)
  local pt   = { x = event.pos.x, z = event.pos.z }
  local side = event.coalition

  local key = normKey(text)
  if STRIKE.FORMATIONS[key] then
    spawnStrikeGroup(STRIKE.FORMATIONS[key], side, pt, event.idx)
    return
  end

  if text:find("CLEAN", 1, true) then
    STRIKE.clearAll()
    if event.idx then trigger.action.removeMark(event.idx) end
    return
  end
end

world.addEventHandler({
  onEvent = function(self, event)
    if event.id == world.event.S_EVENT_MARK_CHANGE then
      onMapMarkerChange(event)
    end
  end
})

trigger.action.outText("StrikeFire script v1 loaded.", 7)
